import asyncio
from datetime import time
import json
import os

from pylog.pylogger import PyLogger

from py_pli.pylib import VUnits
from py_pli.pylib import send_msg

from virtualunits.vu_measurement_unit import VUMeasurementUnit
from virtualunits.meas_seq_generator import meas_seq_generator
from virtualunits.meas_seq_generator import TriggerSignal
from virtualunits.meas_seq_generator import OutputSignal

from urpc_enum.measurementparameter import MeasurementParameter

from fleming.module_test.module_test_util import *

from fleming.common.firmware_util import get_fan_control_endpoint
from fleming.common.firmware_util import get_serial_endpoint


# >>>>>>>NOT TESTED YET<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<

# 2021-03-18/Kay Struebing
# Excitation-Test Prototype 3-6 according to
# https://confluence-ext.perkinelmer.com/pages/viewpage.action?pageId=416913531
# Intention: Module Test for Excitation Functions
#
# Flashlamp fl
# TRF-Laser trf
# Alpha Laser al


# Global Constants
PATH, SCRIPT = os.path.split(__file__)
BASE_NAME = SCRIPT.split('.')[0]


# Flashlamp Functions ##################################################################################################

async def fl_test(duration_s=1, frequency=100, power=0.0):
    """
    Switch on Flashlamp for x secs.

    duration_s: The duration in seconds.
    frequency: The flash frequency in Hz.
    power: The flash power in the range [0.0, 1.0].
    """
    TEST_NAME = 'fl_test'

    measurement_unit: VUMeasurementUnit = VUnits.instance.hal.measurementUnit
    flashes_corse, flashes_fine = divmod((duration_s * frequency), 65536)
    duty_cycle = 0.5
    ontime_us  = int(1 / frequency * 1e6 * duty_cycle)
    offtime_us = int(1 / frequency * 1e6 * (1 - duty_cycle))
    
    await send_msg(json.dumps({'result': f"Flash-Lamp Check, Duration {duration_s} sec, Frequency {frequency} Hz, Power {power}"}))

    await fl_fan(100)
    await measurement_unit.MeasurementFunctions.SetParameter(MeasurementParameter.FlashLampHighPowerEnable, 0, timeout=1)
    await measurement_unit.MeasurementFunctions.SetParameter(MeasurementParameter.FlashLampPower, power, timeout=1)

    op_id = TEST_NAME
    seq_gen = meas_seq_generator()
    if flashes_corse > 0:
        seq_gen.Loop(flashes_corse)
        seq_gen.Loop(65536)
        seq_gen.TimerWaitAndRestart(ontime_us * 100)
        seq_gen.SetSignals(OutputSignal.Flash)
        seq_gen.TimerWaitAndRestart(offtime_us * 100)
        seq_gen.ResetSignals(OutputSignal.Flash)
        seq_gen.LoopEnd()
        seq_gen.LoopEnd()
    if flashes_fine > 0:
        seq_gen.Loop(flashes_corse)
        seq_gen.TimerWaitAndRestart(ontime_us * 100)
        seq_gen.SetSignals(OutputSignal.Flash)
        seq_gen.TimerWaitAndRestart(offtime_us * 100)
        seq_gen.ResetSignals(OutputSignal.Flash)
        seq_gen.LoopEnd()
    seq_gen.Stop(0)

    measurement_unit.ClearOperations()
    await measurement_unit.LoadTriggerSequence(op_id, seq_gen.currSequence)
    await measurement_unit.ExecuteMeasurement(op_id)
    await asyncio.sleep(duration_s)
    await fl_fan(0)
    return f"{TEST_NAME} done"


async def fl_fan(pwr):
    if (pwr < 0) or (pwr > 100):
        raise ValueError(f"pwr must be in the range [0, 100]")

    # Flash Lamp XIP03 -> Fan connected to FMB
    fan = get_fan_control_endpoint('fmb_fan')
    channel = 5
    await fan.SetSpeed(channel, pwr, timeout=1)
    await fan.Enable(channel, (pwr > 0), timeout=1)
    # Flash Lamp XIP04 -> Fan connected to EEF
    fan = get_fan_control_endpoint('eef_fan')
    channel = 0
    await fan.SetSpeed(channel, pwr, timeout=1)
    await fan.Enable(channel, (pwr > 0), timeout=1)
    return f"Flashlamp Fan {pwr}%"


# Alpha Functions ######################################################################################################

async def al_test(alc_start=0.0, alc_stop=1.0, alc_step=0.005):
    """
    Scans alpha laser current, measures internal photodiode and temperature and writes result to file.

    alc_start: The start value for the alpha laser current in the range [0.0, 1.0].
    alc_stop: The stop value for the alpha laser current in the range [0.0, 1.0].
    alc_step: The increment value for the alpha laser current.
    """
    TEST_NAME = 'al_test'
    alpha_delay = 1

    await send_msg(json.dumps({'result': f"Alpha Laser Current Scan, Start: {alc_start}, Stop: {alc_stop}, Step: {alc_step}"}))
    
    await al_settemp()
    await al_setpower(0)
    await al_enable()
    await al_on()
    await asyncio.sleep(alpha_delay)
    file_name = get_datafile(PATH, BASE_NAME, TEST_NAME)
    with open(file_name, 'a') as data:
        data.write(f"alc_start = {alc_start}; alc_stop = {alc_stop}; alc_step = {alc_step}\n")
        data.write("AlphaCurrent;AlphaPD;AlphaTemp\n")
        alc_range = [alc / 1000 for alc in range(round(alc_start * 1000), round(alc_stop * 1000 + 1), round(alc_step * 1000))]
        for alpha_current in alc_range:
            await al_setpower(alpha_current)
            await asyncio.sleep(alpha_delay)
            alpd = await al_getpd()
            altemp = await al_gettemp()
            result = f"{alpha_current:.3f}; {alpd}; {altemp}"
            data.write(result + '\n')
            PyLogger.logger.info(result)
            await send_msg(json.dumps({'result': result}))

    await al_off()
    return f"{TEST_NAME} done, output written to {file_name}"

# TRF Laser Functions
# moved to system_test/trf_test.py